<?php declare(strict_types=1);

namespace MagmodulesWebshopnl\Core\Api\Controller\Order;

use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use MagmodulesWebshopnl\Service\WebShopLogger;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Cart\Price\Struct\CalculatedPrice;
use Shopware\Core\Checkout\Cart\Tax\Struct\CalculatedTaxCollection;
use Shopware\Core\Checkout\Cart\Tax\Struct\TaxRuleCollection;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStates;
use Shopware\Core\Checkout\Order\OrderStates;
use Shopware\Core\Checkout\Order\SalesChannel\OrderService;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Util\Json;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Util\Random;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\HttpFoundation\JsonResponse;

class OrderProcess extends AbstractController
{
    /**
     * @var EntityRepository
     */
    public $orderRepository;

    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var CartService
     */
    private $cartService;

    /**
     * @var EntityRepository
     */
    private $salesChannelRepository;

    /**
     * @var EntityRepository
     */
    private $languageRepository;

    /**
     * @var EntityRepository
     */
    private $stateMachineRepository;

    /**
     * @var EntityRepository
     */
    private $stateMachineStateRepository;

    /**
     * @var SystemConfigService
     */
    private $systemConfigService;

    /**
     * @var EntityRepository
     */
    private $customerRepository;

    /**
     * @var EntityRepository
     */
    private $customerAddressRepository;

    /**
     * @var EntityRepository
     */
    private $paymentRepository;

    /**
     * @var EntityRepository
     */
    private $salutationRepository;

    /**
     * @var EntityRepository
     */
    private $numberRangeRepository;

    /**
     * @var EntityRepository
     */
    private $shippingMethodRepository;

    /**
     * @var LoggerInterface
     */
    public $logger;

    /**
     * @var EntityRepository
     */
    public $logEntryRepository;

    /**
     * @var EntityRepository
     */
    public $numberRangeStateRepository;

    /**
     * @var EntityRepository
     */
    public $webshopnlOrderRepository;

    /**
     * @var WebShopLogger
     */
    public $webShopLogger;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    public function __construct(
        EntityRepository $orderRepository,
        OrderService $orderService,
        CartService $cartService,
        EntityRepository $salesChannelRepository,
        EntityRepository $languageRepository,
        EntityRepository $stateMachineRepository,
        EntityRepository $stateMachineStateRepository,
        SystemConfigService $systemConfigService,
        EntityRepository $customerRepository,
        EntityRepository $customerAddressRepository,
        EntityRepository $paymentRepository,
        EntityRepository $salutationRepository,
        EntityRepository $numberRangeRepository,
        EntityRepository $shippingMethodRepository,
        EntityRepository $numberRangeStateRepository,
        EntityRepository $webshopnlOrderRepository,
        WebShopLogger $webShopLogger,
        EntityRepository $productRepository
    ) {
        $this->orderRepository = $orderRepository;
        $this->orderService = $orderService;
        $this->cartService = $cartService;
        $this->salesChannelRepository = $salesChannelRepository;
        $this->languageRepository = $languageRepository;
        $this->stateMachineRepository = $stateMachineRepository;
        $this->stateMachineStateRepository = $stateMachineStateRepository;
        $this->systemConfigService = $systemConfigService;
        $this->customerRepository = $customerRepository;
        $this->customerAddressRepository = $customerAddressRepository;
        $this->paymentRepository = $paymentRepository;
        $this->salutationRepository = $salutationRepository;
        $this->numberRangeRepository = $numberRangeRepository;
        $this->shippingMethodRepository = $shippingMethodRepository;
        $this->numberRangeStateRepository = $numberRangeStateRepository;
        $this->webshopnlOrderRepository = $webshopnlOrderRepository;
        $this->webShopLogger = $webShopLogger;
        $this->productRepository = $productRepository;
    }

    public function processOrder($orders, $salesChannelId, $context)
    {
        $log = $this->systemConfigService->get('MagmodulesWebshopnl.settings.debug', $salesChannelId);

        foreach ($orders as $order) {
            $checkExistOrder = $this->getCheckExistOrder($order, $context);
            if ($checkExistOrder > 0) {
                if ($log) {
                    $status = "Rejected";
                    $message = "Order already imported";
                    $this->webShopLogger->addOrderLog($context, $this->getLoggerData($order, $status, $message));
                }
            } else {
                $this->orderDataCreate($order ,$salesChannelId, $context);
            }
        }
    }

    public function getCheckExistOrder($order, $context)
    {
        $webShopOrderId = $order['order_id'];
        $existOrderCriteria = new Criteria();
        $existOrderCriteria->addFilter(new EqualsFilter('webShopOrderId', $webShopOrderId));
        return $this->webshopnlOrderRepository->search($existOrderCriteria, $context)->getTotal();
    }

    public function getLoggerData($order, $status, $message)
    {
        return [
            'order_id' => $order['order_id'] ?? null,
            'merchant_id' => $order['merchant_id'] ?? null,
            'status' => $status,
            'message' => $message
        ];
    }
    public function orderDataCreate($getOrder, $salesChannelId, $context): array
    {
        $languageId = $this->getLanguageDetails($context);
        $custEmail = $getOrder['billing_address']['email'];

        $salesChannelData = $this->getSaleChannelDetails($salesChannelId, $context);
        $countryId = $salesChannelData->getCountryId();
        $currencyId = $salesChannelData->getCurrencyId();

        $orderId = Uuid::randomHex();
        $billingAddress = $this->getOrderAddressId($orderId, $getOrder['billing_address'], $countryId, $context);
        $shippingAddress = $this->getOrderAddressId($orderId, $getOrder['billing_address'], $countryId, $context);
        $orderCustomerId = $this->getCustomerId($custEmail, $salesChannelId, $getOrder, $context, $languageId, $countryId, $billingAddress, $shippingAddress);

        $order = [];
        $order['id']                  = $orderId;
        $order['billingAddress']      = $billingAddress;
        $order['shippingAddress']     = $shippingAddress;
        $order['currencyId']          = $currencyId;
        $order['orderNumber']         = (string)($this->getOrderNumber($context)->getState()->getLastValue()+ 1);
        $order['orderCustomer']       = $this->getOrderCustomer($orderId, $orderCustomerId, $getOrder, $context);
        $order['languageId']	      = $languageId;
        $order['salesChannelId']	  = $salesChannelId;
        $order['price']	              = $this->getPrice($getOrder);
        $order['lineItems']           = $this->getLineItem($orderId, $getOrder, $context);
        $order['orderDateTime']       = (new \DateTime())->format(Defaults::STORAGE_DATE_TIME_FORMAT);
        $order['customerId']	      = $orderCustomerId;
        $order['currencyFactor']      = $salesChannelData->getCurrency()->getFactor();
        $order['deepLinkCode']        = Random::getBase64UrlString(32);
        $order['transactions']        = $this->getOrderTransactions($orderId, $getOrder, $context);
        $order['deliveries']          = $this->getOrderDeliveries($orderId, $shippingAddress, $getOrder, $context);
        $order['shippingCosts']       = $this->getShippingCost($getOrder);
        $order['stateId']             = $this->getIdByOrderStateName(OrderStates::STATE_MACHINE, $context);

        if (array_key_exists('shipping_method',$getOrder)) {
            $shippingTotal =  $getOrder['items']['total'] + $getOrder['shipping_method']['cost'];
        } else {
            $shippingTotal = $getOrder['items']['total'];
        }

        if (array_key_exists('paid_at', $getOrder)) {
            $transactionStatePaidId = $this->getIdByTransactionStateName(OrderTransactionStates::STATE_PAID, $context);
            $order['orderTransactionStateId'] = $transactionStatePaidId;
        } else {
            $order['orderTransactionStateId'] = $this->getIdByOrderPaymentStateName($context);
        }

        $order['orderDeliveryStateId'] = $this->getIdByOrderDeliveryStateName($context);
        $order['itemRounding'] = json_decode(Json::encode($salesChannelData->getCurrency()->getItemRounding()), true, 512, \JSON_THROW_ON_ERROR);
        $order['totalRounding'] = json_decode(Json::encode($salesChannelData->getCurrency()->getTotalRounding()), true, 512, \JSON_THROW_ON_ERROR);

        $log = $this->systemConfigService->get('MagmodulesWebshopnl.settings.debug', $salesChannelId);

        try {
            $this->orderRepository->create([$order], $context);
            $this->updateOrderNumber($context);
            $this->addWebShopOrder($order['id'], $getOrder, $context, $salesChannelId);
            if ($log) {
                $status = "Confirmed";
                $message = "Order was imported";
                $this->webShopLogger->addOrderLog($context, $this->getLoggerData($getOrder, $status, $message));
            }
        } catch (UniqueConstraintViolationException $exception) {
            if ($log) {
                $status = "Rejected";
                $message = $exception;
                $this->webShopLogger->addOrderLog($context, $this->getLoggerData($getOrder, $status, $message));
            }
        }
    }

    public function addWebShopOrder($orderId, $getOrder, $context, $salesChannelId)
    {
        $webShopOrder = [
            'id' => Uuid::randomHex(),
            'salesChannelId' => $salesChannelId,
            'webShopOrderId'=> $getOrder['order_id'],
            'status' => 'imported',
            'orderId' => $orderId,
            'merchantId' => $getOrder['merchant_id']
        ];
        $this->webshopnlOrderRepository->create([$webShopOrder], $context);
    }

    public function getOrderDeliveries($orderId, $shippingAddress, $getOrder, $context)
    {
        $shippingDateEarliest = date("Y/m/d");
        $latestDate = strtotime("+4 day", strtotime($shippingDateEarliest));
        $shippingDateLatest = date('Y/m/d', $latestDate);

        return [
            [
                'orderId' => $orderId,
                'shippingOrderAddressId' => $shippingAddress['id'],
                'shippingMethodId' => $this->getShippingMethodId($context),
                'stateId' => $this->getIdByOrderDeliveryStateName($context),
                'trackingCodes' => [],
                'shippingDateEarliest' => $shippingDateEarliest,
                'shippingDateLatest' => $shippingDateLatest,
                'shippingCosts'=> $this->getShippingCost($getOrder)
            ]
        ];
    }

    public function getShippingMethodId($context)
    {
        $paymentCriteria = new Criteria();
        $paymentCriteria->addFilter(new EqualsFilter('name', 'webshopnl'));
        return $this->shippingMethodRepository->search($paymentCriteria, $context)->first()->getId();
    }
    public function getOrderTransactions($orderId, $getOrder, $context)
    {
        return [
            [
                'orderId' => $orderId,
                'paymentMethodId' => $this->getPaymentMethodID($context),
                'amount' =>$this->getTransactionAmount($getOrder),
                'stateId' => $this->getIdByOrderPaymentStateName($context)
            ]
        ];
    }

    public function getPaymentMethodID($context)
    {
        $paymentCriteria = new Criteria();
        $paymentCriteria->addFilter(new EqualsFilter('name', 'webshopnl'));
        return $this->paymentRepository->search($paymentCriteria, $context)->first()->getId();
    }

    public function getTransactionAmount($getOrder) : array
    {
        $quantity =  $getOrder['items']['quantity'];
        $productIds =  $getOrder['items']['products'];
        foreach ($productIds as $data) {
            $price = $data['per_item'];
        }

        return [
            "quantity"=> $quantity,
            "taxRules"=> [
                [
                    "taxRate"=> 0,
                    "percentage"=> 0
                ]
            ],
            "listPrice"=> null,
            "unitPrice"=> $price,
            "totalPrice"=> $price * $quantity,
            "referencePrice"=> null,
            "calculatedTaxes"=> [
                [
                    "tax"=> 0,
                    "price"=> 0,
                    "taxRate"=> 0,
                ]
            ],
            "regulationPrice"=> null
        ];
    }

    public function getLineItem($orderId, $getOrder, $context): array
    {
        $lineItemReturn = [];
        foreach ($getOrder['items']['products'] as $lineItem)
        {
            $getProduct = $this->getProductData($lineItem['remote_id'], $context);
            $lineItemNew =  [
                'id' => Uuid::randomHex(),
                'orderId' => $orderId,
                'productId' => $lineItem['remote_id'],
                'payload'=> [
                    'stock' => $getProduct->getstock(),
                    'productNumber' => $getProduct->getProductNumber(),
                ],
                'label' => $lineItem['name'],
                'quantity' => $lineItem['quantity'],
                'price' => $this->getOrderLineItemPrice($lineItem),
                'removable' => true,
                'stackable' => true,
                'type' => 'product',
                'identifier' => $lineItem['remote_id'],
                'referencedId' => $lineItem['remote_id']
            ];

            array_push($lineItemReturn, $lineItemNew);
        }
        return $lineItemReturn;
    }

    public function getProductData($productId, $context)
    {
        $criteria = new  Criteria();
        $criteria->addFilter(new EqualsFilter('id', $productId));
        return $this->productRepository->search($criteria, $context)->first();
    }
    public function getOrderLineItemPrice($lineItem)
    {
        return [
            "quantity"=> $lineItem['quantity'],
            "taxRules"=> [
                [
                    "taxRate"=> 0,
                    "extensions"=> [],
                    "percentage"=> 0
                ]
            ],
            "listPrice" => null,
            "unitPrice" => $lineItem['per_item'],
            "totalPrice"=>  $lineItem['per_item'] * $lineItem['quantity'],
            "referencePrice"=> null,
            "calculatedTaxes" => [
                [
                    "tax" => 0,
                    "price"=> 0,
                    "taxRate"=> 0,
                    "extensions"=> []
                ]
            ]
        ];
    }

    public function getOrderCustomer($orderId, $orderCustomerId, $getOrder, $context): array
    {
        return [
            'customerId'=> $orderCustomerId,
            'orderId' => $orderId,
            'email' => $getOrder['billing_address']['email'],
            'salutationId' => $this->getValidSalutationId($context),
            'firstName' => $getOrder['billing_address']['first_name'],
            'lastName' =>$getOrder['billing_address']['surname']
        ];
    }

    public function getShippingCost($getOrder)
    {
        return [
                "quantity" => $getOrder['items']['quantity'],
                "taxRules" => [
                    [
                        "taxRate"=> 0,
                        "extensions"=> [],
                        "percentage"=> 0
                    ]
                ],
                "listPrice"=> null,
                "unitPrice"=> 0,
                "totalPrice"=> $getOrder['items']['delivery_cost'],
                "referencePrice"=> null,
                "calculatedTaxes"=> [
                    [
                        "tax"=> 0,
                        "price"=> 0,
                        "taxRate"=> 0,
                        "extensions"=> []
                    ]
                ],
                "regulationPrice"=> null
            ];
    }

    public function getOrderNumber($context)
    {
        $criteria = new  Criteria();
        $criteria->addFilter(new EqualsFilter('name','Orders'));
        $criteria->addAssociation('state');
        return $this->numberRangeRepository->search($criteria, $context)->first();
    }

    public function getPrice($getOrder): array
    {
        $productIds =  $getOrder['items']['products'];
        foreach ($productIds as $data) {
            $price = $data['per_item'];
        }

        return [
            "totalPrice"  => $getOrder['items']['total'],
            "netPrice" => $getOrder['items']['total'],
            "rawTotal" => $getOrder['items']['total'],
            "taxRules" => [
                [
                    "taxRate" => 0,
                    "extensions" => [],
                    "percentage" => 0
                ]
            ],
            "taxStatus" => "gross",
            "positionPrice" => $getOrder['items']['products_cost'],
            "calculatedTaxes" => [
                [
                    "tax" => 0,
                    "price" => 0,
                    "taxRate" => 0,
                    "extensions" => []
                ]
            ]
        ];
    }

    public function updateOrderNumber($context)
    {
        $orderNumberRange = $this->getOrderNumber($context);
        $numberRange = [
            'id' => $orderNumberRange->getState()->getId(),
            'numberRangeId' => $orderNumberRange->getState()->getNumberRangeId(),
            'lastValue' => ($this->getOrderNumber($context)->getState()->getLastValue()+ 1)
        ];
        $this->numberRangeStateRepository->upsert([$numberRange], $context);
    }

    public function getCustomerId($custEmail, $salesChannelId, $getOrder, $context, $languageId, $countryId, $billingAddress, $shippingAddress)
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('email', $custEmail));
        $result = $this->customerRepository->search($criteria, $context)->first();
        if ($result == null) {
            $customerId =  Uuid::randomHex();
            $customerData = [
                'id' => $customerId,
                'groupId' => $this->systemConfigService->get('MagmodulesWebshopnl.settings.defaultCustomerGroup', $salesChannelId),
                'defaultPaymentMethodId' => $this->getValidPaymentMethodId($salesChannelId, $context),
                'salesChannelId' => $salesChannelId,
                'languageId' => $languageId,
                'defaultBillingAddress' => $billingAddress,
                'defaultShippingAddress' => $shippingAddress,
                'customerNumber' => "10142",
                'firstName' => $getOrder['billing_address']['first_name'],
                'lastName' =>$getOrder['billing_address']['surname'],
                'email' => $getOrder['billing_address']['email']
            ];
            $this->customerRepository->create([$customerData], $context);
        } else {
            return $result->getId();
        }
    }

    public function getValidPaymentMethodId($salesChannelId, $context): string
    {
        $criteria = (new Criteria())
            ->setLimit(1)
            ->addFilter(new EqualsFilter('active', true));

        if ($salesChannelId) {
            $criteria->addFilter(new EqualsFilter('salesChannels.id', $salesChannelId));
        }

        /** @var string $id */
        $id = $this->paymentRepository->searchIds($criteria, $context)->firstId();

        return $id;
    }
    public function getIdByTransactionStateName($name, $context)
    {
        $stateMachineCriteria = new Criteria();
        $stateMachineCriteria->addFilter(new EqualsFilter('technicalName', OrderTransactionStates::STATE_PAID));
        return $this->stateMachineStateRepository->search($stateMachineCriteria, $context)->first()->getId();
    }

    public function getIdByOrderStateName($name, $context)
    {
        $stateMachineCriteria = new Criteria();
        $stateMachineCriteria->addFilter(new EqualsFilter('technicalName', OrderStates::STATE_MACHINE));
        return $this->stateMachineRepository->search($stateMachineCriteria, $context)->first()->getInitialStateId();
    }

    public function getSaleChannelDetails($salesChannelId,$context)
    {
        $criteria = new  Criteria();
        $criteria->addAssociation('currency');
        $criteria->addFilter(new EqualsFilter('id',$salesChannelId));
        return $this->salesChannelRepository->search($criteria, $context)->first();
    }

    public function getLanguageDetails($context)
    {
        $criteria = new  Criteria();
        $criteria->addFilter(new EqualsFilter('locale.code','en-GB'));
        return $this->languageRepository->search($criteria, $context)->first()->getId();
    }

    public function getAddressId($customerId, $getOrder, $countryId, $context)
    {
        return [
            'id' => Uuid::randomHex(),
            'customerId' => $customerId,
            'countryId' => $countryId,
            'firstName' => $getOrder['first_name'],
            'lastName' => $getOrder['surname'],
            'zipcode' => $getOrder['zip_code'],
            'city' => $getOrder['city'],
            'street' => $getOrder['street_name'],
            'additionalAddressLine1' => $getOrder['extra_address_information'],
            'created_at' => (new \DateTime())->format(Defaults::STORAGE_DATE_TIME_FORMAT)
        ];
    }

    public function getOrderAddressId($orderId, $getOrder, $countryId, $context)
    {
        return [
            'id' => Uuid::randomHex(),
            'countryId' => $countryId,
            'orderId' => $orderId,
            'salutationId' => $this->getValidSalutationId($context),
            'firstName' => $getOrder['first_name'],
            'lastName' => $getOrder['surname'],
            'street' => $getOrder['street_name'],
            'zipcode' => $getOrder['zip_code'],
            'city' => $getOrder['city'],
            'additionalAddressLine1' => $getOrder['extra_address_information'],
            'created_at' => (new \DateTime())->format(Defaults::STORAGE_DATE_TIME_FORMAT)
        ];
    }

    protected function getValidSalutationId($context): string
    {
        $criteria = (new Criteria())
            ->setLimit(1)
            ->addSorting(new FieldSorting('salutationKey'));

        /** @var string $id */
        $id = $this->salutationRepository->searchIds($criteria, $context)->firstId();

        return $id;
    }

    public function getOrderStatus($orderId, $context)
    {
        $getOrderCriteria = new  Criteria();
        $getOrderCriteria->addFilter(new EqualsFilter('webShopOrderId', $orderId));
        $getOrderCriteria->addAssociation('order');
        $getOrderCriteria->addAssociation('order.transactions');

        $order = $this->webshopnlOrderRepository->search($getOrderCriteria, $context);

        if ($order->getTotal() > 0) {
            return new JsonResponse([
                'id' => $orderId,
                'order status' => $order->first()->getOrder()->getTransactions()->first()->getStateMachineState()->getTechnicalName()
            ]);
        } else {
            return new JsonResponse([
                'status' => "No order found",
            ]);
        }
    }

    public function getIdByOrderDeliveryStateName($context)
    {
        $stateMachineCriteria1 = new Criteria();
        $stateMachineCriteria1->addFilter(new EqualsFilter('technicalName', 'order_delivery.state'));
        return $this->stateMachineRepository->search($stateMachineCriteria1, $context)->first()->getInitialStateId();
    }
    public function getIdByOrderPaymentStateName($context)
    {
        $stateMachineCriteria1 = new Criteria();
        $stateMachineCriteria1->addFilter(new EqualsFilter('technicalName', 'order_transaction.state'));
        return $this->stateMachineRepository->search($stateMachineCriteria1, $context)->first()->getInitialStateId();
    }
}